package com.alipay.gateway.common;

import com.alipay.sofa.gateway.api.domain.ApiConstants;
import com.alipay.sofa.gateway.api.domain.ApiSecretKey;
import com.alipay.sofa.gateway.api.filter.jax.NeedSign;
import com.alipay.sofa.gateway.api.internal.util.ApiLogger;
import com.alipay.sofa.gateway.api.internal.util.StringUtils;
import com.alipay.sofa.gateway.api.internal.util.io.IOUtils;
import com.alipay.sofa.gateway.api.internal.util.sign.jax.JaxSignUtil;
import com.alipay.sofa.gateway.api.internal.util.sign.jax.JaxUtil;
import org.apache.commons.collections.CollectionUtils;

import javax.annotation.Priority;
import javax.ws.rs.Priorities;
import javax.ws.rs.WebApplicationException;
import javax.ws.rs.container.*;
import javax.ws.rs.ext.Provider;
import javax.ws.rs.ext.WriterInterceptor;
import javax.ws.rs.ext.WriterInterceptorContext;
import java.io.*;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

/**
 * @Author 蕖兰
 * @Date 2021/9/10 3:55 下午
 * @Description TODO
 */
@NeedSign
@Provider
@Priority(Priorities.USER)
@PreMatching
public class CustomApiRestSignFilter implements WriterInterceptor, ContainerResponseFilter,
        ContainerRequestFilter {

    private static ThreadLocal<ContainerResponseContext> containerResponseContext = new ThreadLocal<ContainerResponseContext>();
    private static ThreadLocal<ContainerRequestContext> containerRequestContext = new ThreadLocal<ContainerRequestContext>();
    private static ThreadLocal<ApiSecretKey> mathecdApiSecretKey = new ThreadLocal<ApiSecretKey>();

    // 默认开启验签
    private Boolean checkSign;

    /**
     * 秘钥集
     */
    private List<ApiSecretKey> apiSecretKeys;

    /**
     * 匹配url
     */
    private List<String> uriTemplates;

    /**
     * 获取REST请求对象（ThreadLocal）
     *
     * @return property value of containerRequestContext
     */
    public static ContainerRequestContext getContainerRequestContext() {
        return containerRequestContext.get();
    }

    @Override
    public void aroundWriteTo(WriterInterceptorContext context) throws IOException,
            WebApplicationException {
        //思想：先代理输出流，获取输出数据；后拿到输出数据再回写给原有的输出流

        //1. 暂存原有输出流引用
        OutputStream outputStream = context.getOutputStream();

        //2. 指定输出流代理
        ByteArrayOutputStream byteArrayOutputStream = new ByteArrayOutputStream();
        context.setOutputStream(byteArrayOutputStream);

        try {
            //3. 执行写入数据到代理输出流
            context.proceed();
        } finally {
            //4. 获取代理输出流数据
            byte[] bytes = byteArrayOutputStream.toByteArray();

            //5. 响应体签名
            try {
                List<ApiSecretKey> keys;
                if (null == mathecdApiSecretKey.get()) {
                    keys = apiSecretKeys;
                } else {
                    keys = new ArrayList<ApiSecretKey>();
                    keys.add(mathecdApiSecretKey.get());
                }
                JaxSignUtil.sign(containerResponseContext.get(), bytes, keys);
            } catch (Exception ex) {
                ApiLogger.error(
                        "SOFA Gateway 签名过滤器-签名生成异常, " + buildRequestLog(containerRequestContext.get()),
                        ex);
                throw new RuntimeException(ex);
            }

            //7. 将数据回写给原输出流
            outputStream.write(bytes);
            ApiLogger.info(
                    buildResponseLog(containerRequestContext.get(), containerResponseContext.get()));
        }

    }

    @Override
    public void filter(ContainerRequestContext requestContext,
                       ContainerResponseContext responseContext) throws IOException {
        List<ApiSecretKey> keys;
        if (null == mathecdApiSecretKey.get()) {
            keys = apiSecretKeys;
        } else {
            keys = new ArrayList<ApiSecretKey>();
            keys.add(mathecdApiSecretKey.get());
        }
        // 响应加签，body是有数据的，由于从outputstream获取字节流比较困难，不对body加签，也就是签名头里没有digest摘要信息
        JaxSignUtil.sign(responseContext, null, keys);

        containerResponseContext.set(responseContext);
    }

    @Override
    public void filter(ContainerRequestContext requestContext) throws IOException {

        containerRequestContext.set(requestContext);
        if (null == checkSign) {
            String flag = System.getProperty(ApiConstants.CHECK_SIGN_FLAG);
            checkSign = "true".equalsIgnoreCase(flag);
        }
        if (!checkSign) {
            return;
        }

        String uri = requestContext.getUriInfo().getPath();
        if (!matchUri(uri)) {
            return;
        }

        ApiLogger.info(buildRequestLog(requestContext));

        InputStream inputStream = requestContext.getEntityStream();
        // 1. 由于数据已经存储在request的InputStream中，因此先读取出来保存
        byte[] bytes = null;
        if (null != inputStream) {
            bytes = IOUtils.toByteArray(inputStream);
            // 2. 再将读取出来的数据回写到代理的InputStream，用于后面容器返回
            ByteArrayInputStream byteArrayInputStream = new ByteArrayInputStream(bytes);
            requestContext.setEntityStream(byteArrayInputStream);
        }

        // 3. 验签
        try {
            ApiSecretKey apiSecretKey = JaxSignUtil.checkSign(requestContext, bytes, apiSecretKeys);
            mathecdApiSecretKey.set(apiSecretKey);
        } catch (Exception ex) {
            ApiLogger.error("SOFA Gateway 签名过滤器-签名校验异常, " + buildRequestLog(requestContext), ex);
            throw new RuntimeException(ex);
        }
    }

    private boolean matchUri(String uri) {
        if (StringUtils.isEmpty(uri)) {
            return false;
        }
        // 全量匹配
        if (CollectionUtils.isEmpty(uriTemplates)) {
            return true;
        }

        // 遍历模板匹配
        boolean matched = false;
        if (CollectionUtils.isNotEmpty(uriTemplates)) {
            for (String uriTemplate : uriTemplates) {
                if (uri.matches(uriTemplate)) {
                    matched = true;
                    break;
                }
            }
        }
        return matched;
    }

    /**
     * Setter method for property <tt>ApiSecretKeys</tt>.
     *
     * @param apiSecretKeys value to be assigned to property ApiSecretKeys
     */
    public void setApiSecretKeys(List<ApiSecretKey> apiSecretKeys) {
        this.apiSecretKeys = apiSecretKeys;
    }

    private String buildRequestLog(ContainerRequestContext request) {
        Map<String, String> headers = JaxUtil.getHeaders(request);
        String uri = request.getUriInfo().getPath();
        StringBuilder sb = new StringBuilder();
        sb.append("[SOFA Gateway 签名过滤器] ");
        sb.append("uri:[").append(uri).append("];");
        sb.append("request header:[").append(headers).append("];");
        return sb.toString();
    }

    private String buildResponseLog(ContainerRequestContext request,
                                    ContainerResponseContext response) {
        Map<String, String> requestHeaders = JaxUtil.getHeaders(request);
        Map<String, String> responseHeaders = JaxUtil.getHeaders(response);

        String uri = request.getUriInfo().getPath();
        StringBuilder sb = new StringBuilder();
        sb.append("[SOFA Gateway 签名过滤器] ");
        sb.append("uri:[").append(uri).append("];");
        sb.append("request header:[").append(requestHeaders).append("];");
        sb.append("response header:[").append(responseHeaders).append("];");
        return sb.toString();
    }

    public Boolean getCheckSign() {
        return checkSign;
    }

    public void setCheckSign(Boolean checkSign) {
        this.checkSign = checkSign;
    }

    public void setUriTemplates(List<String> uriTemplates) {
        this.uriTemplates = uriTemplates;
    }
}
