//
//  LocalTTSViewController.mm
//  NUIdemo
//
//  Created by zhuiyin on 2020/11/30.
//  All rights reserved.
//

#import "nuisdk.framework/Headers/NeoNuiTts.h"
#import "LocalTTSViewController.h"
#import "HWOptionButton.h"
#import "NuiSdkUtils.h"

#import <mach/mach.h>
#import <assert.h>
//#import <pthread.h>

#import "audio/NLSPlayAudio.h"
#import <AVFoundation/AVFoundation.h>

#import <AdSupport/ASIdentifierManager.h>

#ifdef DEBUG_TTS_DATA_SAVE
FILE * fp;
#endif

#define CPU_USAGE_INTERVAL 0.05
static LocalTTSViewController *myself = nil;
static NSString *myvoicedir = nil;
static NSString *myfontname = nil; // 记录发音人
@interface LocalTTSViewController () <NlsPlayerDelegate, UITextFieldDelegate, HWOptionButtonDelegate, NeoNuiTtsDelegate> {
    IBOutlet UIButton *PlayButton;
    IBOutlet UIButton *TestButton;
    IBOutlet UIButton *PauseButton;
    IBOutlet UITextView *textViewContent;
    IBOutlet UITextView *instructionContent;

    IBOutlet UILabel *labelFontName;
    IBOutlet UILabel *labelSpeedLevel;
    IBOutlet UILabel *labelPitchLevel;
    IBOutlet UILabel *labelVolume;

    IBOutlet UITextField *textfieldSpeedLevel;
    IBOutlet UITextField *textfieldPitchLevel;
    IBOutlet UITextField *textfieldVolume;
    
    NSString * playingContent;
}

@property(nonatomic,strong) NeoNuiTts* nui;
@property(nonatomic, weak) HWOptionButton *fontName;
@property(nonatomic, weak) HWOptionButton *modeType;
@property(nonatomic,strong) NLSPlayAudio *voicePlayer;
@property(nonatomic,strong) NuiSdkUtils *utils;
@end

@implementation LocalTTSViewController

#define SCREEN_WIDTH_BASE 375
#define SCREEN_HEIGHT_BASE 667

static CGSize global_size;
static BOOL initialized = NO;

#pragma mark view controller methods

- (void)viewDidLoad {
    TLog(@"LocalTTSViewController did load");
    [super viewDidLoad];
    self.view.backgroundColor = [UIColor whiteColor];
    self.navigationItem.title = @"本地语音合成";

    // Do any additional setup after loading the view.
    global_size = [UIScreen mainScreen].bounds.size;
    TLog(@"LocalTTSViewController-viewDidLoad mainScreen width=%f  height=%f",
          global_size.width, global_size.height);

    myself = self;
    
    [self InitView];
    // 注意：这里voiceplyer模块仅用于播放示例，未经过大规模线上测试
    _voicePlayer = [[NLSPlayAudio alloc] init];
    _voicePlayer.delegate = self;
    
    _utils = [NuiSdkUtils alloc];
    myvoicedir = [[NSString alloc] init];
    myfontname = [[NSString alloc] init];

    [self NuiTtsInit];
}

-(void)dealloc {
    TLog(@"%s",__FUNCTION__);
    [_voicePlayer cleanup];
    [_nui nui_tts_release];
    initialized = NO;
#ifdef DEBUG_TTS_DATA_SAVE
    if (fp) {
        fclose(fp);
    }
#endif
}

- (void)dismissKeyboard:(id)sender {
    [self.view endEditing:YES];
}

-(void)viewDidAppear:(BOOL)animated {
    TLog(@"LocalTTSViewController-viewDidAppear");
    [super viewDidAppear:animated];
    [self InitView];
}

-(void)viewWillDisappear:(BOOL)animated {
    TLog(@"LocalTTSViewController-viewWillDisappear");
    [_voicePlayer cleanup];
    [_nui nui_tts_release];
}

- (void)didReceiveMemoryWarning {
    [super didReceiveMemoryWarning];
}

#pragma mark - Audio Player Delegate
-(void) playerDidFinish{
    //播放数据被实际播放完成后回调。
    TLog(@"playerDidFinish");
}

#pragma mark -private methods

-(NSString *)genInitParams {
    NSString *strResourcesBundle = [[NSBundle mainBundle] pathForResource:@"Resources" ofType:@"bundle"];
    NSString *bundlePath = [[NSBundle bundleWithPath:strResourcesBundle] resourcePath];
    NSString *id_string = [[[ASIdentifierManager sharedManager] advertisingIdentifier] UUIDString];
    NSString *debug_path = [_utils createDir];
    myvoicedir = debug_path;
    NSMutableDictionary *dictM = [NSMutableDictionary dictionary];
    
    [dictM setObject:bundlePath forKey:@"workspace"];
    // 如果需要保存调试日志到文件，初始化的时候加入该字段；不要保存日志，该字段删除
    // 日志文件是追加的方式存储的，下次初始化时并不会将老日志冲掉重写
    // 另外，在开启保存日志文件时，可以动态的通过接口nui_tts_set_param将日志等级设置成最高值，保证日志不写入文件，等需要写入时再动态设置成较低的等级
    [dictM setObject:debug_path forKey:@"debug_path"];

    TLog(@"debug_path:%@", debug_path);
    TLog(@"workspace:%@", bundlePath);

    // 请参照阿里云官网获取鉴权信息获取配额
    // https://help.aliyun.com/document_detail/251488.html?spm=a2c4g.11186623.6.638.1f0d530eut95Jn
    // 如果配额已耗尽，请联系客户扩大配额
    // 如果合成失败，通常是由于鉴权失败，可以参照阿里云官网Q&A部分
    // https://help.aliyun.com/document_detail/204191.html?spm=a2c4g.11186623.6.657.3cde7340qMll1h ，根据错误日志判别导致鉴权失败的原因

    //请使用您的阿里云账号与appkey进行访问, 以下介绍两种方案
    //方案一(有安全风险,不推荐):
    //  首先ak_id ak_secret app_key如何获得,请查看https://help.aliyun.com/document_detail/72138.html
    //  涉及账户安全,以下账号信息请妥善保存,不可明文存储。
    //  可将以下账号信息加密后存在服务器,待需要传入SDK时在解密;或在本地加密后存储(不推荐,仍然有风险)。
//    [dictM setObject:@"xx" forKey:@"ak_id"]; // 必填
//    [dictM setObject:@"xx" forKey:@"ak_secret"]; // 必填
//    [dictM setObject:@"xx" forKey:@"app_key"]; // 必填
    
    //方案二(推荐):
    //  首先ak_id ak_secret app_key如何获得,请查看https://help.aliyun.com/document_detail/72138.html
    //  然后请看https://help.aliyun.com/document_detail/466615.html
    //  使用其中方案二使用STS获取临时账号
    [dictM setObject:@"STS.xx" forKey:@"ak_id"]; // 必填
    [dictM setObject:@"xx" forKey:@"ak_secret"]; // 必填
    [dictM setObject:@"xx" forKey:@"app_key"]; // 必填
    [dictM setObject:@"xx" forKey:@"sts_token"]; // 必填

    // 离线语音合成sdk_code取值：精品版为software_nls_tts_offline， 标准版为software_nls_tts_offline_standard
    // 离线语音合成账户和sdk_code也可用于唤醒
    [dictM setObject:@"software_nls_tts_offline_standard" forKey:@"sdk_code"]; // 必填

    // deviceid作为设备的唯一标识（可以是Mac地址，cpu序列号等参数），是进行鉴权认证的必要信息
    //   更换手机或者更换device_id都会导致重新鉴权计费。
    //   传入与其他账号的相同device_id或随机变换的device_id会导致鉴权失败或重复收费。
    [dictM setObject:id_string forKey:@"device_id"]; // 必填

    // 设置成本地语音合成模式, 这个设置很重要, 遗漏会导致无法运行
    [dictM setObject:@"0" forKey:@"mode_type"]; // 必填

    NSData *data = [NSJSONSerialization dataWithJSONObject:dictM options:NSJSONWritingPrettyPrinted error:nil];
    NSString * jsonStr = [[NSString alloc]initWithData:data encoding:NSUTF8StringEncoding];
    return jsonStr;
}

- (void)NuiTtsInit {
    if (_nui == NULL) {
        _nui = [NeoNuiTts get_instance];
    }
    NSString * initParam = [self genInitParams];

    // 初始化:需要特别注意该函数并没有加载语音包
    int retcode = [_nui nui_tts_initialize:[initParam UTF8String] logLevel:LOG_LEVEL_VERBOSE saveLog:YES];
    if (retcode != 0) {
        const char *errmsg = [_nui nui_tts_get_param: "error_msg"];
        TLog(@"init failed. retcode:%d. errmsg:%s", retcode, errmsg);
        return;
    } else {
        TLog(@"init success.");
    }

    // 加载语音包：已购买的语音包，可以放在任意位置，以aijia为例，该语音包位于Documents/voices/下，设置命令为“Documents/voices/aijia”
    NSString *cmd = [NSString stringWithFormat:@"%@/aijia", myvoicedir];
    NSFileManager *fileManager = [NSFileManager defaultManager];
    if (![fileManager fileExistsAtPath:cmd]) {
        TLog(@"NuiTtsInit 设置发音人(%@)不存在，尝试使用默认Bundle中发音人", cmd);
        NSString *strResourcesBundle = [[NSBundle mainBundle] pathForResource:@"Resources" ofType:@"bundle"];
        NSString *bundlePath = [[NSBundle bundleWithPath:strResourcesBundle] resourcePath];
        cmd = [NSString stringWithFormat:@"%@/tts/voices/%@", bundlePath, @"aijia"];
    }

    myfontname = cmd;
    retcode = [self.nui nui_tts_set_param:"extend_font_name" value:[cmd UTF8String]];
    // 必须保证语音包设置正确才能正常合成
    if (retcode != SUCCESS) {
        TLog(@"set(%@) failed. the voice package must be set successfully!", cmd);
        const char *errmsg = [_nui nui_tts_get_param: "error_msg"];
        TLog(@"set extend_font_name failed. retcode:%d. errmsg:%s", retcode, errmsg);
        return;
    } else {
        TLog(@"NuiTtsInit set extend_font_name success.");

        const char *sr_s = [_nui nui_tts_get_param: "model_sample_rate"];
        if (sr_s) {
            NSString *sr_string = [NSString stringWithFormat:@"%s", sr_s];
            int sr = [sr_string intValue];
            TLog(@"set sample_rate to %d", sr);
            [_voicePlayer setsamplerate:(int)sr];
        }
    }
    initialized = YES;

#ifdef DEBUG_TTS_DATA_SAVE
    NSString *sp = self.createDir;
    const char* savePath = [sp UTF8String];

    if (fp == nullptr) {
        NSString *debug_file = [NSString stringWithFormat:@"%@/tts_dump.pcm", sp];
        fp = fopen([debug_file UTF8String], "w");
    }
#endif
}

-(void)InitView {
    // init Button
    [self setButton];
    // init TextView
    [self setTextView];
    // init Label
    [self setLabel];
    // init OptionButton
    [self setOptionButton];
    // init TextField
    [self setTextField];
    // init setInstruction
    [self setInstruction];
}

- (void)setButton {
    CGFloat button_width = global_size.width/SCREEN_WIDTH_BASE * 80;
    CGFloat button_height = global_size.height/SCREEN_HEIGHT_BASE * 40;
    CGFloat x = global_size.width/SCREEN_WIDTH_BASE * 27.5;
    CGFloat y = global_size.height/SCREEN_HEIGHT_BASE * 65;
    
    PlayButton = [UIButton buttonWithType:UIButtonTypeCustom];
    PlayButton.frame = CGRectMake(x, y, button_width, button_height);
    UIImage *image = [UIImage imageNamed:@"button_start"];
    [PlayButton setBackgroundImage:image forState:UIControlStateNormal];
    [PlayButton setTitle:@"播放" forState:UIControlStateNormal];
    [PlayButton setTitleColor:UIColor.blackColor forState:UIControlStateNormal];
    PlayButton.titleLabel.font = [UIFont systemFontOfSize:18];
    [PlayButton addTarget:self action:@selector(startTTS:) forControlEvents:UIControlEventTouchUpInside];
    [self.view addSubview:PlayButton];


    // ---- PauseButton ---
    CGFloat PauseButton_width = global_size.width/SCREEN_WIDTH_BASE * 80;
    CGFloat PauseButton_height = global_size.height/SCREEN_HEIGHT_BASE * 40;
    CGFloat PauseButton_x = (global_size.width - PauseButton_width)/2;
    CGFloat PauseButton_y = global_size.height/SCREEN_HEIGHT_BASE * 65;
    
    PauseButton = [UIButton buttonWithType:UIButtonTypeCustom];
    PauseButton.frame = CGRectMake(PauseButton_x, PauseButton_y, PauseButton_width, PauseButton_height);
    UIImage *PauseButton_image = [UIImage imageNamed:@"button_start"];
    [PauseButton setBackgroundImage:PauseButton_image forState:UIControlStateNormal];
    [PauseButton setTitle:@"暂停" forState:UIControlStateNormal];
    [PauseButton setTitleColor:UIColor.blackColor forState:UIControlStateNormal];
    PauseButton.titleLabel.font = [UIFont systemFontOfSize:18];
    [PauseButton addTarget:self action:@selector(pauseTTS:) forControlEvents:UIControlEventTouchUpInside];
    [self.view addSubview:PauseButton];
}

- (void)setTextView {
    // ---- textViewContent ---
    CGFloat textViewContent_width = global_size.width/SCREEN_WIDTH_BASE * 320;
    CGFloat textViewContent_height = global_size.height/SCREEN_HEIGHT_BASE * 220;
    CGFloat textViewContent_x = global_size.width/2 - textViewContent_width/2;
    CGFloat textViewContent_y = global_size.height/SCREEN_HEIGHT_BASE * 80 + global_size.height/SCREEN_HEIGHT_BASE * 30;
    
    CGRect textViewContent_rect = CGRectMake(textViewContent_x, textViewContent_y, textViewContent_width, textViewContent_height);
    if (!textViewContent) {
        textViewContent = [[UITextView alloc] initWithFrame:textViewContent_rect];
    }
    textViewContent.layer.borderWidth = 0.6;
    textViewContent.layer.borderColor = [UIColor blackColor].CGColor;
    textViewContent.layer.cornerRadius = 10;
    [textViewContent setBackgroundColor: [UIColor colorWithRed:0/255.0f green:0/255.0f blue:0/255.0f alpha:0.1]];
    textViewContent.scrollEnabled = YES;

    textViewContent.text = @"离线语音合成，通过先进的深度学习技术，将文本转换成自然流畅的语音。目前有多种音色可供选择，并提供调节语速、语调、音量等功能。适用于文学有声阅读和无障碍播报等场景。";
    textViewContent.textColor = [UIColor darkGrayColor];
    textViewContent.font = [UIFont systemFontOfSize:15];
    [self.view addSubview:textViewContent];
}

- (void)setLabel {
    // ---- labelFontName ---
    CGFloat labelFontName_width = global_size.width/SCREEN_WIDTH_BASE * 180;
    CGFloat labelFontName_height = global_size.height/SCREEN_HEIGHT_BASE * 40;
    CGFloat labelFontName_x = global_size.width/SCREEN_WIDTH_BASE * 27.5;
    CGFloat labelFontName_y = global_size.height/SCREEN_HEIGHT_BASE * (315 + 15);

    CGRect labelFontName_rect = CGRectMake(labelFontName_x, labelFontName_y, labelFontName_width, labelFontName_height);
    labelFontName = [[UILabel alloc] initWithFrame:labelFontName_rect];
    labelFontName.text = @"font name:";
    labelFontName.textColor = [UIColor blackColor];
    labelFontName.backgroundColor = [UIColor whiteColor];
    labelFontName.font = [UIFont boldSystemFontOfSize:15];
    [self.view addSubview:labelFontName];

    // ---- labelSpeedLevel ---
    CGFloat labelSpeedLevel_width = global_size.width/SCREEN_WIDTH_BASE * 180;
    CGFloat labelSpeedLevel_height = global_size.height/SCREEN_HEIGHT_BASE * 40;
    CGFloat labelSpeedLevel_x = global_size.width/SCREEN_WIDTH_BASE * 27.5;
    CGFloat labelSpeedLevel_y = global_size.height/SCREEN_HEIGHT_BASE * (355 + 15);

    CGRect labelSpeedLevel_rect = CGRectMake(labelSpeedLevel_x, labelSpeedLevel_y, labelSpeedLevel_width, labelSpeedLevel_height);
    labelSpeedLevel = [[UILabel alloc] initWithFrame:labelSpeedLevel_rect];
    labelSpeedLevel.text = @"speed level(0~2):";
    labelSpeedLevel.textColor = [UIColor blackColor];
    labelSpeedLevel.backgroundColor = [UIColor whiteColor];
    labelSpeedLevel.font = [UIFont boldSystemFontOfSize:15];
    [self.view addSubview:labelSpeedLevel];

    // ---- labelPitchLevel ---
    CGFloat labelPitchLevel_width = global_size.width/SCREEN_WIDTH_BASE * 180;
    CGFloat labelPitchLevel_height = global_size.height/SCREEN_HEIGHT_BASE * 40;
    CGFloat labelPitchLevel_x = global_size.width/SCREEN_WIDTH_BASE * 27.5;
    CGFloat labelPitchLevel_y = global_size.height/SCREEN_HEIGHT_BASE * (395 + 15);

    CGRect labelPitchLevel_rect = CGRectMake(labelPitchLevel_x, labelPitchLevel_y, labelPitchLevel_width, labelPitchLevel_height);
    labelPitchLevel = [[UILabel alloc] initWithFrame:labelPitchLevel_rect];
    labelPitchLevel.text = @"pitch level(-500~500):";
    labelPitchLevel.textColor = [UIColor blackColor];
    labelPitchLevel.backgroundColor = [UIColor whiteColor];
    labelPitchLevel.font = [UIFont boldSystemFontOfSize:15];
    [self.view addSubview:labelPitchLevel];

    // ---- labelVolume ---
    CGFloat labelVolume_width = global_size.width/SCREEN_WIDTH_BASE * 180;
    CGFloat labelVolume_height = global_size.height/SCREEN_HEIGHT_BASE * 40;
    CGFloat labelVolume_x = global_size.width/SCREEN_WIDTH_BASE * 27.5;
    CGFloat labelVolume_y = global_size.height/SCREEN_HEIGHT_BASE * (435 + 15);

    CGRect labelVolume_rect = CGRectMake(labelVolume_x, labelVolume_y, labelVolume_width, labelVolume_height);
    labelVolume = [[UILabel alloc] initWithFrame:labelVolume_rect];
    labelVolume.text = @"volume(0~2):";
    labelVolume.textColor = [UIColor blackColor];
    labelVolume.backgroundColor = [UIColor whiteColor];
    labelVolume.font = [UIFont boldSystemFontOfSize:15];
    [self.view addSubview:labelVolume];
}

- (void)setInstruction {
    TLog(@"setInstruction");
    // ---- instruction ---
    CGFloat instructionViewContent_width = global_size.width/SCREEN_WIDTH_BASE * 340;
    CGFloat instructionViewContent_height = global_size.height/SCREEN_HEIGHT_BASE * 140;
    CGFloat instructionViewContent_x = global_size.width/2 - instructionViewContent_width/2;
//    CGFloat instructionViewContent_y = 530;
    // below the label of volume
    CGFloat instructionViewContent_y = global_size.height/SCREEN_HEIGHT_BASE * (435 + 15) + global_size.height/SCREEN_HEIGHT_BASE * 40 + 10;
    
    CGRect instruction_rect = CGRectMake(instructionViewContent_x, instructionViewContent_y, instructionViewContent_width, instructionViewContent_height);
    if (!instructionContent) {
        instructionContent = [[UITextView alloc] initWithFrame:instruction_rect];
    }
    instructionContent.layer.borderWidth = 0.6;
    instructionContent.layer.borderColor = [UIColor blackColor].CGColor;
    instructionContent.layer.cornerRadius = 10;
    [instructionContent setBackgroundColor: [UIColor colorWithRed:0/255.0f green:0/255.0f blue:0/255.0f alpha:0.1]];
    instructionContent.scrollEnabled = YES;

    instructionContent.text = @"说明：1)demo需要在函数genInitParams设置鉴权信息；2)SDK和语音包是隔离的，SDK不包含语音包，在使用Demo进行播放的时候，请先将语音包推到应用沙盒路径/Documents/voices目录下，点击‘请选择’按钮选择语音包，否则demo无法播报；";
    instructionContent.textColor = [UIColor darkGrayColor];
    instructionContent.font = [UIFont systemFontOfSize:15];
    [self.view addSubview:instructionContent];
}

- (void)setTextField {
    // ---- textfieldSpeedLevel ---
    CGFloat textfieldSpeedLevel_width = global_size.width/SCREEN_WIDTH_BASE * 110;
    CGFloat textfieldSpeedLevel_height = global_size.height/SCREEN_HEIGHT_BASE * 40;
    CGFloat textfieldSpeedLevel_x = global_size.width/SCREEN_WIDTH_BASE * 210;
    CGFloat textfieldSpeedLevel_y = global_size.height/SCREEN_HEIGHT_BASE * (355 + 15);

    CGRect textfieldSpeedLevel_rect = CGRectMake(textfieldSpeedLevel_x, textfieldSpeedLevel_y, textfieldSpeedLevel_width, textfieldSpeedLevel_height);
    textfieldSpeedLevel = [[UITextField alloc] initWithFrame:textfieldSpeedLevel_rect];
    textfieldSpeedLevel.borderStyle = UITextBorderStyleRoundedRect;
    textfieldSpeedLevel.font = [UIFont fontWithName:@"Arial" size:15];
    textfieldSpeedLevel.textColor = [UIColor blackColor];
    textfieldSpeedLevel.backgroundColor = [UIColor lightGrayColor];
    textfieldSpeedLevel.userInteractionEnabled = YES;
    [self.view addSubview:textfieldSpeedLevel];

    // ---- textfieldPitchLevel ---
    CGFloat textfieldPitchLevel_width = global_size.width/SCREEN_WIDTH_BASE * 110;
    CGFloat textfieldPitchLevel_height = global_size.height/SCREEN_HEIGHT_BASE * 40;
    CGFloat textfieldPitchLevel_x = global_size.width/SCREEN_WIDTH_BASE * 210;
    CGFloat textfieldPitchLevel_y = global_size.height/SCREEN_HEIGHT_BASE * (395 + 15);

    CGRect textfieldPitchLevel_rect = CGRectMake(textfieldPitchLevel_x, textfieldPitchLevel_y, textfieldPitchLevel_width, textfieldPitchLevel_height);
    textfieldPitchLevel = [[UITextField alloc] initWithFrame:textfieldPitchLevel_rect];
    textfieldPitchLevel.borderStyle = UITextBorderStyleRoundedRect;
    textfieldPitchLevel.font = [UIFont fontWithName:@"Arial" size:15];
    textfieldPitchLevel.textColor = [UIColor blackColor];
    textfieldPitchLevel.backgroundColor = [UIColor lightGrayColor];
    textfieldPitchLevel.userInteractionEnabled = YES;
    [self.view addSubview:textfieldPitchLevel];

    // ---- textfieldVolume ---
    CGFloat textfieldVolume_width = global_size.width/SCREEN_WIDTH_BASE * 110;
    CGFloat textfieldVolume_height = global_size.height/SCREEN_HEIGHT_BASE * 40;
    CGFloat textfieldVolume_x = global_size.width/SCREEN_WIDTH_BASE * 210;
    CGFloat textfieldVolume_y = global_size.height/SCREEN_HEIGHT_BASE * (435 + 15);

    CGRect textfieldVolume_rect = CGRectMake(textfieldVolume_x, textfieldVolume_y, textfieldVolume_width, textfieldVolume_height);
    textfieldVolume = [[UITextField alloc] initWithFrame:textfieldVolume_rect];
    textfieldVolume.borderStyle = UITextBorderStyleRoundedRect;
    textfieldVolume.font = [UIFont fontWithName:@"Arial" size:15];
    textfieldVolume.textColor = [UIColor blackColor];
    textfieldVolume.backgroundColor = [UIColor lightGrayColor];
    textfieldVolume.userInteractionEnabled = YES;
    [self.view addSubview:textfieldVolume];
}

- (void)setOptionButton {
    // ---- fontName ---
    CGFloat fontName_width = global_size.width/SCREEN_WIDTH_BASE * 150;
    CGFloat fontName_height = global_size.height/SCREEN_HEIGHT_BASE * 40;
    CGFloat fontName_x = global_size.width/SCREEN_WIDTH_BASE * (SCREEN_WIDTH_BASE - 27.5 - 150);
    CGFloat fontName_y = global_size.height/SCREEN_HEIGHT_BASE * 95 + global_size.height/SCREEN_HEIGHT_BASE * 220 + global_size.height/SCREEN_HEIGHT_BASE * 15;

    HWOptionButton *fontNameBtn = [[HWOptionButton alloc] initWithFrame:CGRectMake(fontName_x, fontName_y, fontName_width, fontName_height)];
    fontNameBtn.backgroundColor = [UIColor whiteColor];
    fontNameBtn.array = @[@"aijia"];
    fontNameBtn.delegate = self;
    fontNameBtn.showSearchBar = YES;
    [self.view addSubview:fontNameBtn];
    self.fontName = fontNameBtn;

}

-(void) UpdateTtsParams {
    if ([_fontName.title isEqualToString:@"-请选择-"]) {
        // DO NOTHING
        TLog(@"do nothing ... 请首先选择有效路径（如/Documents/voices）下的语音包");
    } else if ([_fontName.title isEqualToString:@"aijia"]) {
        // 切换语音包，如准备切换到aijia，aijia语音包位于Documents/voices/下，切换命令为“Documents/voices/aijia”
        // 需要注意的是：该接口有一定计算量，建议只有在发音人发生变化时调用
        NSString *cmd = [NSString stringWithFormat:@"%@/%@", myvoicedir, _fontName.title];
        NSFileManager *fileManager = [NSFileManager defaultManager];
        if (![fileManager fileExistsAtPath:cmd]) {
            TLog(@"发音人(%@)不存在，尝试使用默认Bundle中发音人", cmd);
            NSString *strResourcesBundle = [[NSBundle mainBundle] pathForResource:@"Resources" ofType:@"bundle"];
            NSString *bundlePath = [[NSBundle bundleWithPath:strResourcesBundle] resourcePath];
            cmd = [NSString stringWithFormat:@"%@/tts/voices/%@", bundlePath, @"aijia"];
        }
        if ([myfontname isEqualToString:cmd]) {
            TLog(@"发音人没有发生变化，不需要切换");
        } else {
            int retcode = [self.nui nui_tts_set_param:"extend_font_name" value:[cmd UTF8String]];
            if (retcode != SUCCESS) {
                TLog(@"switch(%@) failed, continue to use the original voice", cmd);
            } else {
                TLog(@"UpdateTtsParams(%@) success", cmd);
                myfontname = cmd;

                const char *sr_s = [_nui nui_tts_get_param: "model_sample_rate"];
                if (sr_s) {
                    NSString *sr_string = [NSString stringWithFormat:@"%s", sr_s];
                    int sr = [sr_string intValue];
                    TLog(@"set sample_rate to %d", sr);
                    [_voicePlayer setsamplerate:(int)sr];
                }
            }
        }
        initialized = YES;
    } else {
        NSString *cmd = [NSString stringWithFormat:@"%@/%@", myvoicedir, _fontName.title];
        if ([myfontname isEqualToString:cmd]) {
            TLog(@"发音人没有发生变化，不需要切换");
        } else {
            int retcode = [self.nui nui_tts_set_param:"extend_font_name" value:[cmd UTF8String]];
            if (retcode != SUCCESS) {
                TLog(@"switch(%@) failed, continue to use the original voice", cmd);
            } else {
                myfontname = cmd;
            }
        }
        initialized = YES;
    }
    if (self->textfieldSpeedLevel.text.length > 0) {
        // 设置语速，默认1.0
        [self.nui nui_tts_set_param:"speed_level" value:[textfieldSpeedLevel.text UTF8String]];
    }
    if (self->textfieldPitchLevel.text.length > 0) {
        // 设置音调，默认0
        [self.nui nui_tts_set_param:"pitch_level" value:[textfieldPitchLevel.text UTF8String]];
    }
    if (self->textfieldVolume.text.length > 0) {
        // 设置音量，默认1.0
        [self.nui nui_tts_set_param:"volume" value:[textfieldVolume.text UTF8String]];
    }
}

#pragma mark - Button Action
- (IBAction)startTTS:(UIButton *)sender {
    playingContent = textViewContent.text;
    if (!self.nui) {
        TLog(@"tts not init");
        return;
    }
    
    NSString *content = textViewContent.text;
    dispatch_async(dispatch_get_main_queue(), ^{ 
        [self UpdateTtsParams];
        AVAudioSession *audioSession = [AVAudioSession sharedInstance];
        [audioSession setCategory:AVAudioSessionCategoryPlayAndRecord withOptions:AVAudioSessionCategoryOptionDefaultToSpeaker error:nil];
        if (initialized) {
            // 如果上个任务没有合成完毕，手动取消，开始合成新的任务
            [self.nui nui_tts_cancel:NULL];

            [self.nui nui_tts_play:"1" taskId:"" text:[content UTF8String]];
        } else {
            TLog(@"It has not been initialized!!!");
        }
    });

    dispatch_async(dispatch_get_main_queue(), ^{
        UIImage *image = [UIImage imageNamed:@"button_stop"];
        [PlayButton setBackgroundImage:image forState:UIControlStateNormal];
        [PlayButton setTitle:@"停止" forState:UIControlStateNormal];
        [PlayButton removeTarget:self action:@selector(startTTS:) forControlEvents:UIControlEventTouchUpInside];
        [PlayButton addTarget:self action:@selector(stopTTS:) forControlEvents:UIControlEventTouchUpInside];
    });
}

- (IBAction)stopTTS:(UIButton *)sender {
    if (_nui != nil) {
        TLog(@"LocalTTSViewController stop tts");
        dispatch_async(dispatch_get_main_queue(), ^{
            [self.nui nui_tts_cancel:NULL];
            [self->_voicePlayer stop];
        });
        
        dispatch_async(dispatch_get_main_queue(), ^{
            // UI更新代码
            UIImage *image = [UIImage imageNamed:@"button_start"];
            [PlayButton setBackgroundImage:image forState:UIControlStateNormal];
            [PlayButton setTitle:@"播放" forState:UIControlStateNormal];
            [PlayButton removeTarget:self action:@selector(stopTTS:) forControlEvents:UIControlEventTouchUpInside];
            [PlayButton addTarget:self action:@selector(startTTS:) forControlEvents:UIControlEventTouchUpInside];
            
            [PauseButton setBackgroundImage:image forState:UIControlStateNormal];
            [PauseButton setTitle:@"暂停" forState:UIControlStateNormal];
            [PauseButton removeTarget:self action:@selector(resumeTTS:) forControlEvents:UIControlEventTouchUpInside];
            [PauseButton addTarget:self action:@selector(pauseTTS:) forControlEvents:UIControlEventTouchUpInside];
        });
    } else {
        TLog(@"in stopTTS, _nui == nil.");
    }
}


- (IBAction)pauseTTS:(UIButton *)sender {
    if (_nui != nil) {
        dispatch_async(dispatch_get_main_queue(), ^{
            [self.nui nui_tts_pause];
            [self.voicePlayer pause];
            // UI更新代码
            UIImage *image = [UIImage imageNamed:@"button_stop"];
            [PauseButton setBackgroundImage:image forState:UIControlStateNormal];
            [PauseButton setTitle:@"继续" forState:UIControlStateNormal];
            [PauseButton removeTarget:self action:@selector(pauseTTS:) forControlEvents:UIControlEventTouchUpInside];
            [PauseButton addTarget:self action:@selector(resumeTTS:) forControlEvents:UIControlEventTouchUpInside];
        });
    } else {
        TLog(@"in pauseTTS, _nui == nil.");
    }
}

- (IBAction)resumeTTS:(UIButton *)sender {
    if (_nui != nil) {
        dispatch_async(dispatch_get_main_queue(), ^{
            [self.nui nui_tts_resume];
            [self.voicePlayer resume];
            // UI更新代码
            UIImage *image = [UIImage imageNamed:@"button_start"];
            [PauseButton setBackgroundImage:image forState:UIControlStateNormal];
            [PauseButton setTitle:@"暂停" forState:UIControlStateNormal];
            [PauseButton removeTarget:self action:@selector(resumeTTS:) forControlEvents:UIControlEventTouchUpInside];
            [PauseButton addTarget:self action:@selector(pauseTTS:) forControlEvents:UIControlEventTouchUpInside];
        });
    } else {
        TLog(@"in resumeTTS, _nui == nil.");
    }
}

#pragma mark - tts callback
- (void)onNuiTtsEventCallback:(NuiSdkTtsEvent)event taskId:(char*)taskid code:(int)code {
    TLog(@"onNuiTtsEventCallback event[%d]", event);
    if (event == TTS_EVENT_START) {
        // 启动播放器
        [self->_voicePlayer play];

    } else if (event == TTS_EVENT_END || event == TTS_EVENT_CANCEL || event == TTS_EVENT_ERROR) {
        if (event == TTS_EVENT_END) {
            // 注意这里是指语音合成完成，而非播放完成，播放完成需要由voicePlayer对象来进行通知
            [self->_voicePlayer drain];
        } else {
            // 取消播报、或者发生异常时终止播放
            [self->_voicePlayer stop];
        }
        if (event == TTS_EVENT_ERROR) {
            const char *errmsg = [_nui nui_tts_get_param: "error_msg"];
            TLog(@"tts get errmsg:%s", errmsg);
        }
        dispatch_async(dispatch_get_main_queue(), ^{
            // UI更新代码
            UIImage *image = [UIImage imageNamed:@"button_start"];
            [PlayButton setBackgroundImage:image forState:UIControlStateNormal];
            [PlayButton setTitle:@"播放" forState:UIControlStateNormal];
            [PlayButton removeTarget:self action:@selector(stopTTS:) forControlEvents:UIControlEventTouchUpInside];
            [PlayButton addTarget:self action:@selector(startTTS:) forControlEvents:UIControlEventTouchUpInside];
        });
    }
}

- (void)onNuiTtsUserdataCallback:(char*)info infoLen:(int)info_len buffer:(char*)buffer len:(int)len taskId:(char*)task_id {
    TLog(@"onNuiTtsUserdataCallback info text %s index %d len:%d", info, info_len, len);
    if (len > 0) {
        [_voicePlayer write:(char*)buffer Length:(unsigned int)len];
    }
}

-(void)onNuiTtsVolumeCallback:(int)volume taskId:(char*)task_id {
    ;
}

@end
