//
//  TTSViewController.mm
//  NUIdemo
//
//  Created by zhouguangdong on 2019/12/20.
//  Copyright © 2019年 zhouguangdong. All rights reserved.
//
//#define DEBUG_MODE
//#define DEBUG_TTS_DATA_SAVE
#import "nuisdk.framework/Headers/NeoNuiTts.h"
#import "TTSViewController.h"
#import "HWOptionButton.h"
#import "NuiSdkUtils.h"

#import <mach/mach.h>
#import <assert.h>
#import <pthread.h>

#import "audio/NLSPlayAudio.h"
#import <AVFoundation/AVFoundation.h>

#import <AdSupport/ASIdentifierManager.h>

#ifdef DEBUG_TTS_DATA_SAVE
FILE * fp;
#endif

#define CPU_USAGE_INTERVAL 0.05
static TTSViewController *myself = nil;

// 本样例展示在线语音合成使用方法
// iOS SDK 详细说明：https://help.aliyun.com/document_detail/173947.html
@interface TTSViewController () <NlsPlayerDelegate, UITextFieldDelegate, HWOptionButtonDelegate, NeoNuiTtsDelegate> {
    IBOutlet UIButton *PlayButton;
    IBOutlet UIButton *TestButton;
    IBOutlet UIButton *PauseButton;
    IBOutlet UITextView *textViewContent;

    IBOutlet UILabel *labelFontName;
    IBOutlet UILabel *labelModeType;
    IBOutlet UILabel *labelSpeedLevel;
    IBOutlet UILabel *labelPitchLevel;
    IBOutlet UILabel *labelVolume;

    IBOutlet UITextField *textfieldSpeedLevel;
    IBOutlet UITextField *textfieldPitchLevel;
    IBOutlet UITextField *textfieldVolume;
    
    NSString * playingContent;
}

@property(nonatomic,strong) NeoNuiTts* nui;
@property(nonatomic, weak) HWOptionButton *fontName;
@property(nonatomic, weak) HWOptionButton *modeType;
@property(nonatomic,strong) NLSPlayAudio *voicePlayer;
@property(nonatomic,strong) NuiSdkUtils *utils;
@end

@implementation TTSViewController

#define SCREEN_WIDTH_BASE 375
#define SCREEN_HEIGHT_BASE 667

static CGSize global_size;
static int loop_in = 0;
static BOOL loop_flag = NO;

#pragma mark view controller methods

- (void)viewDidLoad {
    TLog(@"TTSViewController did load");
    [super viewDidLoad];
    self.view.backgroundColor = [UIColor whiteColor];
    self.navigationItem.title = @"语音合成";

    // Do any additional setup after loading the view.
    global_size = [UIScreen mainScreen].bounds.size;
    TLog(@"TTSViewController-viewDidLoad mainScreen width=%f  height=%f",
          global_size.width, global_size.height);

    loop_in = TTS_EVENT_END;
    myself = self;
    
    // 这里voiceplyer模块仅用于示例（可根据实际音频采样率在NLSPlayAudio.mm中进行更改，如改为16000/24000Hz）
    [self InitView];
    _voicePlayer = [[NLSPlayAudio alloc] init];
    _voicePlayer.delegate = self;
    
    _utils = [NuiSdkUtils alloc];

    [self NuiTtsInit];
}

-(void)dealloc {
    TLog(@"%s",__FUNCTION__);
    [_voicePlayer cleanup];
    [_nui nui_tts_release];
#ifdef DEBUG_TTS_DATA_SAVE
    if (fp) {
        fclose(fp);
    }
#endif
}

- (void)dismissKeyboard:(id)sender {
    [self.view endEditing:YES];
}

-(void)viewDidAppear:(BOOL)animated {
    TLog(@"TTSViewController-viewDidAppear");
    [super viewDidAppear:animated];
    [self InitView];
}

-(void)viewWillDisappear:(BOOL)animated {
    TLog(@"TTSViewController-viewWillDisappear");
    [NSObject cancelPreviousPerformRequestsWithTarget:self selector:@selector(tts_test_loop:) object:testDataList];
}

- (void)didReceiveMemoryWarning {
    [super didReceiveMemoryWarning];
}

#pragma mark - Audio Player Delegate
-(void) playerDidFinish{
    //播放数据被实际播放完成后回调。
    TLog(@"playerDidFinish");
}

#pragma mark -private methods

-(NSString *)genInitParams {
    NSString *strResourcesBundle = [[NSBundle mainBundle] pathForResource:@"Resources" ofType:@"bundle"];
    NSString *bundlePath = [[NSBundle bundleWithPath:strResourcesBundle] resourcePath];
    NSString *id_string = [[[ASIdentifierManager sharedManager] advertisingIdentifier] UUIDString];
    NSString *debug_path = [_utils createDir];
    NSMutableDictionary *dictM = [NSMutableDictionary dictionary];
    
    [dictM setObject:bundlePath forKey:@"workspace"]; // 必填, 且需要有读写权限
    [dictM setObject:debug_path forKey:@"debug_path"];
    [dictM setObject:id_string forKey:@"device_id"]; // 必填
    
    TLog(@"debug_path:%@", debug_path);
    TLog(@"workspace:%@", bundlePath);
    
    //  首先app_key如何获得,请查看https://help.aliyun.com/document_detail/72138.html
    [dictM setObject:@"xx" forKey:@"app_key"]; // 必填

    //从阿里云获取appkey和token进行语音服务访问
    //请输入您申请的id与token，否则无法使用语音服务，获取方式请参考阿里云官网文档：
    //https://help.aliyun.com/document_detail/72153.html?spm=a2c4g.11186623.6.555.59bd69bb6tkTSc

    //获取token方式一般有两种：
    //方法1：
    //参考如下实现在端上访问阿里云Token服务获取SDK进行获取
    //[_utils getTicket:dictM];

    //方法2：
    //在您的服务端进行token管理，此处获取服务端的token进行语音服务访问
    [dictM setObject:@"xx" forKey:@"token"]; // 必填

    [dictM setObject:@"wss://nls-gateway.cn-shanghai.aliyuncs.com:443/ws/v1" forKey:@"url"]; // 默认
    
    NSData *data = [NSJSONSerialization dataWithJSONObject:dictM options:NSJSONWritingPrettyPrinted error:nil];
    NSString * jsonStr = [[NSString alloc]initWithData:data encoding:NSUTF8StringEncoding];
    return jsonStr;
}

- (void)NuiTtsInit {
    if (_nui == NULL) {
        _nui = [NeoNuiTts get_instance];
        _nui.delegate = self;
    }
    //请注意此处的参数配置，其中账号相关需要在Utils.m getTicket 方法中填入后才可访问服务
    NSString * initParam = [self genInitParams];

    int retcode = [_nui nui_tts_initialize:[initParam UTF8String] logLevel:LOG_LEVEL_VERBOSE saveLog:YES];
    if (retcode != 0) {
         TLog(@"init failed.retcode:%d", retcode);
         return;
     }
#ifdef DEBUG_TTS_DATA_SAVE
    NSString *sp = self.createDir;
    const char* savePath = [sp UTF8String];

    if (fp == nullptr) {
        NSString *debug_file = [NSString stringWithFormat:@"%@/tts_dump.pcm", sp];
        fp = fopen([debug_file UTF8String], "w");
    }
#endif
}

-(void)InitView {
    // init Button
    [self setButton];
    // init TextView
    [self setTextView];
    // init Label
    [self setLabel];
    // init OptionButton
    [self setOptionButton];
    // init TextField
    [self setTextField];
}

- (void)setButton {
    CGFloat button_width = global_size.width/SCREEN_WIDTH_BASE * 80;
    CGFloat button_height = global_size.height/SCREEN_HEIGHT_BASE * 40;
    CGFloat x = global_size.width/SCREEN_WIDTH_BASE * 27.5;
    CGFloat y = global_size.height/SCREEN_HEIGHT_BASE * 65;
    
    PlayButton = [UIButton buttonWithType:UIButtonTypeCustom];
    PlayButton.frame = CGRectMake(x, y, button_width, button_height);
    UIImage *image = [UIImage imageNamed:@"button_start"];
    [PlayButton setBackgroundImage:image forState:UIControlStateNormal];
    [PlayButton setTitle:@"播放" forState:UIControlStateNormal];
    [PlayButton setTitleColor:UIColor.blackColor forState:UIControlStateNormal];
    PlayButton.titleLabel.font = [UIFont systemFontOfSize:18];
    [PlayButton addTarget:self action:@selector(startTTS:) forControlEvents:UIControlEventTouchUpInside];
    [self.view addSubview:PlayButton];

    // ---- TestButton ---
    CGFloat TestButton_width = global_size.width/SCREEN_WIDTH_BASE * 80;
    CGFloat TestButton_height = global_size.height/SCREEN_HEIGHT_BASE * 40;
    CGFloat TestButton_x = global_size.width/SCREEN_WIDTH_BASE * (SCREEN_WIDTH_BASE - 27.5 - 80);
    CGFloat TestButton_y = global_size.height/SCREEN_HEIGHT_BASE * 65;
    
    TestButton = [UIButton buttonWithType:UIButtonTypeCustom];
    TestButton.frame = CGRectMake(TestButton_x, TestButton_y, TestButton_width, TestButton_height);
    UIImage *TestButton_image = [UIImage imageNamed:@"button_start"];
    [TestButton setBackgroundImage:TestButton_image forState:UIControlStateNormal];
    [TestButton setTitle:@"测试" forState:UIControlStateNormal];
    [TestButton setTitleColor:UIColor.blackColor forState:UIControlStateNormal];
    TestButton.titleLabel.font = [UIFont systemFontOfSize:18];
    [TestButton addTarget:self action:@selector(startTest:) forControlEvents:UIControlEventTouchUpInside];
    [self.view addSubview:TestButton];

    // ---- PauseButton ---
    CGFloat PauseButton_width = global_size.width/SCREEN_WIDTH_BASE * 80;
    CGFloat PauseButton_height = global_size.height/SCREEN_HEIGHT_BASE * 40;
    CGFloat PauseButton_x = (global_size.width - PauseButton_width)/2;
    CGFloat PauseButton_y = global_size.height/SCREEN_HEIGHT_BASE * 65;
    
    PauseButton = [UIButton buttonWithType:UIButtonTypeCustom];
    PauseButton.frame = CGRectMake(PauseButton_x, PauseButton_y, PauseButton_width, PauseButton_height);
    UIImage *PauseButton_image = [UIImage imageNamed:@"button_start"];
    [PauseButton setBackgroundImage:PauseButton_image forState:UIControlStateNormal];
    [PauseButton setTitle:@"暂停" forState:UIControlStateNormal];
    [PauseButton setTitleColor:UIColor.blackColor forState:UIControlStateNormal];
    PauseButton.titleLabel.font = [UIFont systemFontOfSize:18];
    [PauseButton addTarget:self action:@selector(pauseTTS:) forControlEvents:UIControlEventTouchUpInside];
    [self.view addSubview:PauseButton];
}

- (void)setTextView {
    // ---- textViewContent ---
    CGFloat textViewContent_width = global_size.width/SCREEN_WIDTH_BASE * 320;
    CGFloat textViewContent_height = global_size.height/SCREEN_HEIGHT_BASE * 220;
    CGFloat textViewContent_x = global_size.width/2 - textViewContent_width/2;
    CGFloat textViewContent_y = global_size.height/SCREEN_HEIGHT_BASE * 80 + global_size.height/SCREEN_HEIGHT_BASE * 30;
    
    CGRect textViewContent_rect = CGRectMake(textViewContent_x, textViewContent_y, textViewContent_width, textViewContent_height);
    if (!textViewContent) {
        textViewContent = [[UITextView alloc] initWithFrame:textViewContent_rect];
    }
    textViewContent.layer.borderWidth = 0.6;
    textViewContent.layer.borderColor = [UIColor blackColor].CGColor;
    textViewContent.layer.cornerRadius = 10;
    [textViewContent setBackgroundColor: [UIColor colorWithRed:0/255.0f green:0/255.0f blue:0/255.0f alpha:0.1]];
    textViewContent.scrollEnabled = YES;

    // 支持一次性合成300字符以内的文字，其中1个汉字、1个英文字母或1个标点均算作1个字符，
    // 超过300个字符的内容将会截断。所以请确保传入的text小于300字符(不包含ssml格式)。
    textViewContent.text = @"语音合成服务，通过先进的深度学习技术，将文本转换成自然流畅的语音。目前有多种音色可供选择，并提供调节语速、语调、音量等功能。适用于智能客服、语音交互、文学有声阅读和无障碍播报等场景。";
    textViewContent.textColor = [UIColor darkGrayColor];
    textViewContent.font = [UIFont systemFontOfSize:15];
    [self.view addSubview:textViewContent];
}

- (void)setLabel {
    // ---- labelFontName ---
    CGFloat labelFontName_width = global_size.width/SCREEN_WIDTH_BASE * 180;
    CGFloat labelFontName_height = global_size.height/SCREEN_HEIGHT_BASE * 40;
    CGFloat labelFontName_x = global_size.width/SCREEN_WIDTH_BASE * 27.5;
    CGFloat labelFontName_y = global_size.height/SCREEN_HEIGHT_BASE * (315 + 15);

    CGRect labelFontName_rect = CGRectMake(labelFontName_x, labelFontName_y, labelFontName_width, labelFontName_height);
    labelFontName = [[UILabel alloc] initWithFrame:labelFontName_rect];
    labelFontName.text = @"font name:";
    labelFontName.textColor = [UIColor blackColor];
    labelFontName.backgroundColor = [UIColor whiteColor];
    labelFontName.font = [UIFont boldSystemFontOfSize:15];
    [self.view addSubview:labelFontName];

    // ---- labelModeType ---
    CGFloat labelModeType_width = global_size.width/SCREEN_WIDTH_BASE * 180;
    CGFloat labelModeType_height = global_size.height/SCREEN_HEIGHT_BASE * 40;
    CGFloat labelModeType_x = global_size.width/SCREEN_WIDTH_BASE * 27.5;
    CGFloat labelModeType_y = global_size.height/SCREEN_HEIGHT_BASE * (355 + 15);

    CGRect labelModeType_rect = CGRectMake(labelModeType_x, labelModeType_y, labelModeType_width, labelModeType_height);
    labelModeType = [[UILabel alloc] initWithFrame:labelModeType_rect];
    labelModeType.text = @"mode type:";
    labelModeType.textColor = [UIColor blackColor];
    labelModeType.backgroundColor = [UIColor whiteColor];
    labelModeType.font = [UIFont boldSystemFontOfSize:15];
    [self.view addSubview:labelModeType];

    // ---- labelSpeedLevel ---
    CGFloat labelSpeedLevel_width = global_size.width/SCREEN_WIDTH_BASE * 180;
    CGFloat labelSpeedLevel_height = global_size.height/SCREEN_HEIGHT_BASE * 40;
    CGFloat labelSpeedLevel_x = global_size.width/SCREEN_WIDTH_BASE * 27.5;
    CGFloat labelSpeedLevel_y = global_size.height/SCREEN_HEIGHT_BASE * (395 + 15);

    CGRect labelSpeedLevel_rect = CGRectMake(labelSpeedLevel_x, labelSpeedLevel_y, labelSpeedLevel_width, labelSpeedLevel_height);
    labelSpeedLevel = [[UILabel alloc] initWithFrame:labelSpeedLevel_rect];
    labelSpeedLevel.text = @"speed level(0~2):";
    labelSpeedLevel.textColor = [UIColor blackColor];
    labelSpeedLevel.backgroundColor = [UIColor whiteColor];
    labelSpeedLevel.font = [UIFont boldSystemFontOfSize:15];
    [self.view addSubview:labelSpeedLevel];

    // ---- labelPitchLevel ---
    CGFloat labelPitchLevel_width = global_size.width/SCREEN_WIDTH_BASE * 180;
    CGFloat labelPitchLevel_height = global_size.height/SCREEN_HEIGHT_BASE * 40;
    CGFloat labelPitchLevel_x = global_size.width/SCREEN_WIDTH_BASE * 27.5;
    CGFloat labelPitchLevel_y = global_size.height/SCREEN_HEIGHT_BASE * (435 + 15);

    CGRect labelPitchLevel_rect = CGRectMake(labelPitchLevel_x, labelPitchLevel_y, labelPitchLevel_width, labelPitchLevel_height);
    labelPitchLevel = [[UILabel alloc] initWithFrame:labelPitchLevel_rect];
    labelPitchLevel.text = @"pitch level(-500~500):";
    labelPitchLevel.textColor = [UIColor blackColor];
    labelPitchLevel.backgroundColor = [UIColor whiteColor];
    labelPitchLevel.font = [UIFont boldSystemFontOfSize:15];
    [self.view addSubview:labelPitchLevel];

    // ---- labelVolume ---
    CGFloat labelVolume_width = global_size.width/SCREEN_WIDTH_BASE * 180;
    CGFloat labelVolume_height = global_size.height/SCREEN_HEIGHT_BASE * 40;
    CGFloat labelVolume_x = global_size.width/SCREEN_WIDTH_BASE * 27.5;
    CGFloat labelVolume_y = global_size.height/SCREEN_HEIGHT_BASE * (475 + 15);

    CGRect labelVolume_rect = CGRectMake(labelVolume_x, labelVolume_y, labelVolume_width, labelVolume_height);
    labelVolume = [[UILabel alloc] initWithFrame:labelVolume_rect];
    labelVolume.text = @"volume(0~2):";
    labelVolume.textColor = [UIColor blackColor];
    labelVolume.backgroundColor = [UIColor whiteColor];
    labelVolume.font = [UIFont boldSystemFontOfSize:15];
    [self.view addSubview:labelVolume];
}

- (void)setTextField {
    // ---- textfieldSpeedLevel ---
    CGFloat textfieldSpeedLevel_width = global_size.width/SCREEN_WIDTH_BASE * 110;
    CGFloat textfieldSpeedLevel_height = global_size.height/SCREEN_HEIGHT_BASE * 40;
    CGFloat textfieldSpeedLevel_x = global_size.width/SCREEN_WIDTH_BASE * 210;
    CGFloat textfieldSpeedLevel_y = global_size.height/SCREEN_HEIGHT_BASE * (395 + 15);

    CGRect textfieldSpeedLevel_rect = CGRectMake(textfieldSpeedLevel_x, textfieldSpeedLevel_y, textfieldSpeedLevel_width, textfieldSpeedLevel_height);
    textfieldSpeedLevel = [[UITextField alloc] initWithFrame:textfieldSpeedLevel_rect];
    textfieldSpeedLevel.borderStyle = UITextBorderStyleRoundedRect;
    textfieldSpeedLevel.font = [UIFont fontWithName:@"Arial" size:15];
    textfieldSpeedLevel.textColor = [UIColor blackColor];
    textfieldSpeedLevel.backgroundColor = [UIColor lightGrayColor];
    textfieldSpeedLevel.userInteractionEnabled = YES;
    [self.view addSubview:textfieldSpeedLevel];

    // ---- textfieldPitchLevel ---
    CGFloat textfieldPitchLevel_width = global_size.width/SCREEN_WIDTH_BASE * 110;
    CGFloat textfieldPitchLevel_height = global_size.height/SCREEN_HEIGHT_BASE * 40;
    CGFloat textfieldPitchLevel_x = global_size.width/SCREEN_WIDTH_BASE * 210;
    CGFloat textfieldPitchLevel_y = global_size.height/SCREEN_HEIGHT_BASE * (435 + 15);

    CGRect textfieldPitchLevel_rect = CGRectMake(textfieldPitchLevel_x, textfieldPitchLevel_y, textfieldPitchLevel_width, textfieldPitchLevel_height);
    textfieldPitchLevel = [[UITextField alloc] initWithFrame:textfieldPitchLevel_rect];
    textfieldPitchLevel.borderStyle = UITextBorderStyleRoundedRect;
    textfieldPitchLevel.font = [UIFont fontWithName:@"Arial" size:15];
    textfieldPitchLevel.textColor = [UIColor blackColor];
    textfieldPitchLevel.backgroundColor = [UIColor lightGrayColor];
    textfieldPitchLevel.userInteractionEnabled = YES;
    [self.view addSubview:textfieldPitchLevel];

    // ---- textfieldVolume ---
    CGFloat textfieldVolume_width = global_size.width/SCREEN_WIDTH_BASE * 110;
    CGFloat textfieldVolume_height = global_size.height/SCREEN_HEIGHT_BASE * 40;
    CGFloat textfieldVolume_x = global_size.width/SCREEN_WIDTH_BASE * 210;
    CGFloat textfieldVolume_y = global_size.height/SCREEN_HEIGHT_BASE * (475 + 15);

    CGRect textfieldVolume_rect = CGRectMake(textfieldVolume_x, textfieldVolume_y, textfieldVolume_width, textfieldVolume_height);
    textfieldVolume = [[UITextField alloc] initWithFrame:textfieldVolume_rect];
    textfieldVolume.borderStyle = UITextBorderStyleRoundedRect;
    textfieldVolume.font = [UIFont fontWithName:@"Arial" size:15];
    textfieldVolume.textColor = [UIColor blackColor];
    textfieldVolume.backgroundColor = [UIColor lightGrayColor];
    textfieldVolume.userInteractionEnabled = YES;
    [self.view addSubview:textfieldVolume];
}

- (void)setOptionButton {
    // ---- fontName ---
    // 在线语音合成发音人可以参考阿里云官网
    // https://help.aliyun.com/document_detail/84435.html
    CGFloat fontName_width = global_size.width/SCREEN_WIDTH_BASE * 150;
    CGFloat fontName_height = global_size.height/SCREEN_HEIGHT_BASE * 40;
    CGFloat fontName_x = global_size.width/SCREEN_WIDTH_BASE * (SCREEN_WIDTH_BASE - 27.5 - 150);
    CGFloat fontName_y = global_size.height/SCREEN_HEIGHT_BASE * 95 + global_size.height/SCREEN_HEIGHT_BASE * 220 + global_size.height/SCREEN_HEIGHT_BASE * 15;

    HWOptionButton *fontNameBtn = [[HWOptionButton alloc] initWithFrame:CGRectMake(fontName_x, fontName_y, fontName_width, fontName_height)];
    fontNameBtn.backgroundColor = [UIColor whiteColor];
    fontNameBtn.array = @[@"xiaoyun", @"xiaogang", @"siqi"];
    fontNameBtn.delegate = self;
    fontNameBtn.showSearchBar = YES;
    [self.view addSubview:fontNameBtn];
    self.fontName = fontNameBtn;
    
    // ---- modeType ---
    CGFloat modeType_width = global_size.width/SCREEN_WIDTH_BASE * 150;
    CGFloat modeType_height = global_size.height/SCREEN_HEIGHT_BASE * 40;
    CGFloat modeType_x = global_size.width/SCREEN_WIDTH_BASE * (SCREEN_WIDTH_BASE - 27.5 - 150);
    CGFloat modeType_y = global_size.height/SCREEN_HEIGHT_BASE * 315 + global_size.height/SCREEN_HEIGHT_BASE * 40 + global_size.height/SCREEN_HEIGHT_BASE * 15;
    
    HWOptionButton *modeTypeBtn = [[HWOptionButton alloc] initWithFrame:CGRectMake(modeType_x, modeType_y, modeType_width, modeType_height)];
    modeTypeBtn.array = @[@"在线合成"];
    modeTypeBtn.backgroundColor = [UIColor whiteColor];
    modeTypeBtn.delegate = self;
    modeTypeBtn.showSearchBar = YES;
    [self.view addSubview:modeTypeBtn];
    self.modeType = modeTypeBtn;
}

- (void)playTTS:(NSString *)content {
    playingContent = content;
    
    if (!self.nui) {
        TLog(@"nui tts not init");
        return;
    }
    
    dispatch_async(dispatch_get_main_queue(), ^{
        textViewContent.text = content;
        [self UpdateTtsParams:content];
        // taskid用户可以自己设置，格式为32字节的uuid，如“bf2e88aa42a743028315ffc0dcb53cda”
        // taskid为空时: SDK内部将自动产生32字节的uuid作为taskid
        [self.nui nui_tts_play:"1" taskId:"" text:[content UTF8String]];
    });
}

-(void) UpdateTtsParams:(NSString *)content {
    if ([_fontName.title isEqualToString:@"-请选择-"]) {
        // DO NOTHING
    } else {
        [self.nui nui_tts_set_param:"font_name" value:[_fontName.title UTF8String]];
    }
    int chars = 0;
    if ([_modeType.title isEqualToString:@"离线合成"]) {
        [self.nui nui_tts_set_param:"mode_type" value:"0"]; // 必填
    } else if ([_modeType.title isEqualToString:@"在线合成"]) {
        [self.nui nui_tts_set_param:"mode_type" value:"2"]; // 必填
        
        // 支持一次性合成300字符以内的文字，其中1个汉字、1个英文字母或1个标点均算作1个字符，
        // 超过300个字符的内容将会截断。所以请确保传入的text小于300字符(不包含ssml格式)。
        // 长短文本语音合成收费不同，请注意。
        // 300字这个截点可能会变更，以官网文档为准。
        chars = [self.nui nui_tts_get_num_of_chars: [content UTF8String]];
        if (chars > 300) {
            // 超过300字符设置成 长文本语音合成 模式
            [self.nui nui_tts_set_param:"tts_version" value:"1"];
        } else {
            // 未超过300字符设置成 短文本语音合成 模式
            [self.nui nui_tts_set_param:"tts_version" value:"0"];
        }
    } else {
        [self.nui nui_tts_set_param:"mode_type" value:"2"]; // 必填

        chars = [self.nui nui_tts_get_num_of_chars: [content UTF8String]];
        if (chars > 300) {
            [self.nui nui_tts_set_param:"tts_version" value:"1"];
        } else {
            [self.nui nui_tts_set_param:"tts_version" value:"0"];
        }
    }

    // 详细参数可见: https://help.aliyun.com/document_detail/173642.html
    if (self->textfieldSpeedLevel.text.length > 0) {
        [self.nui nui_tts_set_param:"speed_level" value:[textfieldSpeedLevel.text UTF8String]];
    }
    if (self->textfieldPitchLevel.text.length > 0) {
        [self.nui nui_tts_set_param:"pitch_level" value:[textfieldPitchLevel.text UTF8String]];
    }
    if (self->textfieldVolume.text.length > 0) {
        [self.nui nui_tts_set_param:"volume)" value:[textfieldVolume.text UTF8String]];
    }
    [self.nui nui_tts_set_param:"enable_subtitle" value:"1"];
}

static NSArray *testDataList = nil;
- (void)tts_test_loop:(NSArray<NSString *>*)list {
    static int i = 0;
    if (!list || i >= list.count) {
        TLog(@"tts test loop finish or list = nil");
        return;
    }
    
    if (loop_in == TTS_EVENT_START) {
        [self performSelector:@selector(tts_test_loop:) withObject:list afterDelay:3];
        return ;
    }
    
    if (loop_in == TTS_EVENT_CANCEL ||
        loop_in == TTS_EVENT_ERROR) {
        TLog(@"Tts canceled or Error");
        return ;
    }
    
    NSString * dialog = list[i];
    [self playTTS:dialog];
    i++;
    
    [self performSelector:@selector(tts_test_loop:) withObject:list afterDelay:3];
}

#pragma mark - Button Action
- (IBAction)startTTS:(UIButton *)sender {
    playingContent = textViewContent.text;
    if (!self.nui) {
        TLog(@"tts not init");
        return;
    }
    
    NSString *content = textViewContent.text;
    dispatch_async(dispatch_get_main_queue(), ^{ 
        [self UpdateTtsParams:content];
        AVAudioSession *audioSession = [AVAudioSession sharedInstance];
        [audioSession setCategory:AVAudioSessionCategoryPlayAndRecord withOptions:AVAudioSessionCategoryOptionDefaultToSpeaker error:nil];
        // 如果上个任务没有合成完毕，手动取消，开始合成新的任务
        [self.nui nui_tts_cancel:NULL];

        [self.nui nui_tts_play:"1" taskId:"" text:[content UTF8String]];
    });

    dispatch_async(dispatch_get_main_queue(), ^{
        UIImage *image = [UIImage imageNamed:@"button_stop"];
        [PlayButton setBackgroundImage:image forState:UIControlStateNormal];
        [PlayButton setTitle:@"停止" forState:UIControlStateNormal];
        [PlayButton removeTarget:self action:@selector(startTTS:) forControlEvents:UIControlEventTouchUpInside];
        [PlayButton addTarget:self action:@selector(stopTTS:) forControlEvents:UIControlEventTouchUpInside];
    });
}

- (IBAction)stopTTS:(UIButton *)sender {
    if (_nui != nil) {
        TLog(@"TTSViewController stop tts");

        dispatch_async(dispatch_get_main_queue(), ^{
            [self.nui nui_tts_cancel:NULL];
            [self->_voicePlayer stop];
        });
        
        dispatch_async(dispatch_get_main_queue(), ^{
            // UI更新代码
            UIImage *image = [UIImage imageNamed:@"button_start"];
            [PlayButton setBackgroundImage:image forState:UIControlStateNormal];
            [PlayButton setTitle:@"播放" forState:UIControlStateNormal];
            [PlayButton removeTarget:self action:@selector(stopTTS:) forControlEvents:UIControlEventTouchUpInside];
            [PlayButton addTarget:self action:@selector(startTTS:) forControlEvents:UIControlEventTouchUpInside];
            [PauseButton setBackgroundImage:image forState:UIControlStateNormal];
            [PauseButton setTitle:@"暂停" forState:UIControlStateNormal];
            [PauseButton removeTarget:self action:@selector(resumeTTS:) forControlEvents:UIControlEventTouchUpInside];
            [PauseButton addTarget:self action:@selector(pauseTTS:) forControlEvents:UIControlEventTouchUpInside];

        });
    } else {
        TLog(@"in stopTTS, _nui == nil.");
    }
}


- (IBAction)pauseTTS:(UIButton *)sender {
    if (_nui != nil) {
        dispatch_async(dispatch_get_main_queue(), ^{
            [self.nui nui_tts_pause];
            [self.voicePlayer pause];
            // UI更新代码
            UIImage *image = [UIImage imageNamed:@"button_stop"];
            [PauseButton setBackgroundImage:image forState:UIControlStateNormal];
            [PauseButton setTitle:@"继续" forState:UIControlStateNormal];
            [PauseButton removeTarget:self action:@selector(pauseTTS:) forControlEvents:UIControlEventTouchUpInside];
            [PauseButton addTarget:self action:@selector(resumeTTS:) forControlEvents:UIControlEventTouchUpInside];
        });
    } else {
        TLog(@"in pauseTTS, _nui == nil.");
    }
}

- (IBAction)resumeTTS:(UIButton *)sender {
    if (_nui != nil) {
        dispatch_async(dispatch_get_main_queue(), ^{
            [self.nui nui_tts_resume];
            [self.voicePlayer resume];
            // UI更新代码
            UIImage *image = [UIImage imageNamed:@"button_start"];
            [PauseButton setBackgroundImage:image forState:UIControlStateNormal];
            [PauseButton setTitle:@"暂停" forState:UIControlStateNormal];
            [PauseButton removeTarget:self action:@selector(resumeTTS:) forControlEvents:UIControlEventTouchUpInside];
            [PauseButton addTarget:self action:@selector(pauseTTS:) forControlEvents:UIControlEventTouchUpInside];
        });
    } else {
        TLog(@"in resumeTTS, _nui == nil.");
    }
}


- (IBAction)startTest:(UIButton *)sender {
    TLog(@"start a pthread for tts test.");
    [NSObject cancelPreviousPerformRequestsWithTarget:self selector:@selector(tts_test_loop:) object:testDataList];

    if (!self.nui) {
        TLog(@"nui tts not init");
    }
    
    loop_flag = YES;
    dispatch_async(dispatch_get_main_queue(), ^{
        UIImage *image = [UIImage imageNamed:@"button_stop"];
        [TestButton setBackgroundImage:image forState:UIControlStateNormal];
        [TestButton setTitle:@"结束" forState:UIControlStateNormal];
        [TestButton removeTarget:self action:@selector(startTest:) forControlEvents:UIControlEventTouchUpInside];
        [TestButton addTarget:self action:@selector(stopTest:) forControlEvents:UIControlEventTouchUpInside];
    });
    
    
    if (!testDataList) {
        TLog(@"get test list");
        NSString *filePath = [[NSBundle mainBundle] pathForResource:@"thirdparty/test" ofType:@"txt"];
        NSString *dataFile = [NSString stringWithContentsOfFile:filePath encoding:NSUTF8StringEncoding error:nil];
        testDataList = [dataFile componentsSeparatedByString:@"\n"];
    }
    
    [self tts_test_loop:testDataList];
    return;
}

- (IBAction)stopTest:(UIButton *)sender {
    [NSObject cancelPreviousPerformRequestsWithTarget:self selector:@selector(tts_test_loop:) object:testDataList];
    
    if (_nui != nil) {
        dispatch_async(dispatch_get_main_queue(), ^{
            [_nui nui_tts_cancel:NULL];
        });
        loop_flag = NO;
        loop_in = TTS_EVENT_END;
        
        dispatch_async(dispatch_get_main_queue(), ^{
            // UI更新代码
            UIImage *image = [UIImage imageNamed:@"button_start"];
            [TestButton setBackgroundImage:image forState:UIControlStateNormal];
            [TestButton setTitle:@"测试" forState:UIControlStateNormal];
            [TestButton removeTarget:self action:@selector(stopTest:) forControlEvents:UIControlEventTouchUpInside];
            [TestButton addTarget:self action:@selector(startTest:) forControlEvents:UIControlEventTouchUpInside];
        });
    } else {
        TLog(@"in stopTest, _nui == nil.");
    }
}


#pragma mark - tts callback
- (void)onNuiTtsEventCallback:(NuiSdkTtsEvent)event taskId:(char*)taskid code:(int)code {
    TLog(@"onNuiTtsEventCallback event[%d]", event);
    if (event == TTS_EVENT_START) {
        loop_in = TTS_EVENT_START;
        // 启动播放器
        [self->_voicePlayer play];
    } else if (event == TTS_EVENT_END || event == TTS_EVENT_CANCEL || event == TTS_EVENT_ERROR) {
        loop_in = event;
        if (event == TTS_EVENT_END) {
            // 注意这里的event事件是指语音合成完成，而非播放完成，播放完成需要由voicePlayer对象来进行通知
            [self->_voicePlayer drain];
        } else {
            // 取消播报、或者发生异常时终止播放
            [self->_voicePlayer stop];
        }
        if (event == TTS_EVENT_ERROR) {
            const char *errmsg = [_nui nui_tts_get_param: "error_msg"];
            TLog(@"tts get errmsg:%s", errmsg);
        }
        dispatch_async(dispatch_get_main_queue(), ^{
            // UI更新代码
            UIImage *image = [UIImage imageNamed:@"button_start"];
            [PlayButton setBackgroundImage:image forState:UIControlStateNormal];
            [PlayButton setTitle:@"播放" forState:UIControlStateNormal];
            [PlayButton removeTarget:self action:@selector(stopTTS:) forControlEvents:UIControlEventTouchUpInside];
            [PlayButton addTarget:self action:@selector(startTTS:) forControlEvents:UIControlEventTouchUpInside];
        });
    }
}

- (void)onNuiTtsUserdataCallback:(char*)info infoLen:(int)info_len buffer:(char*)buffer len:(int)len taskId:(char*)task_id {
    TLog(@"onNuiTtsUserdataCallback info ...");
    if (info_len > 0) {
        TLog(@"onNuiTtsUserdataCallback info text %s. index %d.", info, info_len);
    }
    if (len > 0) {
        [_voicePlayer write:(char*)buffer Length:(unsigned int)len];
    }
}

-(void)onNuiTtsVolumeCallback:(int)volume taskId:(char*)task_id {
    ;
}
@end
